// controllers/reportesController.js
const { poolPromise, sql } = require("../config/db");
const ExcelJS = require("exceljs");
const PDFDocument = require("pdfkit");

// ======================================================
// 📌 1. OBTENER REPORTE (DATOS REALES)
// ======================================================
exports.getReporte = async (req, res) => {
  try {
    const { tipo, desde, hasta } = req.query;

    const pool = await poolPromise;
    let query = "";

    switch (tipo) {
      case "incidentes":
        query = `
          SELECT 
            descripcion, 
            impacto, 
            estado, 
            FORMAT(fecha_registro, 'dd/MM/yyyy') AS fecha
          FROM Incidentes
          WHERE fecha_registro BETWEEN @desde AND @hasta
          ORDER BY fecha_registro DESC
        `;
        break;

      case "caidas":
        query = `
          SELECT 
            a.nombre_activo AS activo,
            FORMAT(c.fecha_inicio, 'dd/MM/yyyy HH:mm') AS inicio,
            FORMAT(c.fecha_fin, 'dd/MM/yyyy HH:mm') AS fin,
            c.duracion_min,
            c.causa,
            c.estado
          FROM Caidas c
          INNER JOIN Activos a ON a.id_activo = c.id_activo
          WHERE c.fecha_inicio BETWEEN @desde AND @hasta
          ORDER BY c.fecha_inicio DESC
        `;
        break;

      case "simulacros":
        query = `
          SELECT
            s.id_simulacro,
            s.tipo_simulacro,
            FORMAT(s.fecha_programada, 'dd/MM/yyyy') AS fecha,
            s.resultado,
            s.observaciones,
            r.nombre_responsable,
            r.tipo,
            r.correo
          FROM Simulacros s
          INNER JOIN Responsables r ON r.id_responsable = s.responsable_id
          WHERE s.fecha_programada BETWEEN @desde AND @hasta
          ORDER BY s.fecha_programada DESC
        `;
        break;

      default:
        return res.status(400).json({ error: "Tipo de reporte inválido" });
    }

    const result = await pool.request()
      .input("desde", sql.DateTime, new Date(desde))
      .input("hasta", sql.DateTime, new Date(hasta))
      .query(query);

    res.json({
      ok: true,
      tipo,
      total: result.recordset.length,
      data: result.recordset,
    });

  } catch (error) {
    console.error("🔥 ERROR en getReporte:", error);
    res.status(500).send("Error al generar reporte");
  }
};


// ======================================================
// 📌 2. EXPORTAR PDF (DATOS REALES)
// ======================================================
exports.exportarPDF = async (req, res) => {
  try {
    const { tipo, desde, hasta } = req.query;
    const pool = await poolPromise;

    let query = "";

    if (tipo === "incidentes") {
      query = `
        SELECT descripcion, impacto, estado, fecha_registro
        FROM Incidentes
        WHERE fecha_registro BETWEEN @desde AND @hasta
      `;
    } else if (tipo === "caidas") {
      query = `
        SELECT c.*, a.nombre_activo
        FROM Caidas c
        INNER JOIN Activos a ON a.id_activo = c.id_activo
        WHERE fecha_inicio BETWEEN @desde AND @hasta
      `;
    } else if (tipo === "simulacros") {
      query = `
        SELECT *
        FROM Simulacros
        WHERE fecha_programada BETWEEN @desde AND @hasta
      `;
    } else {
      return res.status(400).send("Tipo inválido");
    }

    const result = await pool.request()
      .input("desde", sql.DateTime, new Date(desde))
      .input("hasta", sql.DateTime, new Date(hasta))
      .query(query);

    const doc = new PDFDocument();
    res.setHeader("Content-Type", "application/pdf");
    res.setHeader("Content-Disposition", `attachment; filename=reporte_${tipo}.pdf`);

    doc.fontSize(20).text(`Reporte de ${tipo.toUpperCase()}`, { align: "center" });
    doc.moveDown();

    result.recordset.forEach((row, i) => {
      doc.fontSize(11).text(`${i + 1}. ${JSON.stringify(row)}`);
      doc.moveDown(0.5);
    });

    doc.end();
    doc.pipe(res);
  } catch (error) {
    console.error("🔥 ERROR en exportarPDF:", error);
    res.status(500).send("Error al exportar PDF");
  }
};


// ======================================================
// 📌 3. EXPORTAR EXCEL (DATOS REALES)
// ======================================================
exports.exportarExcel = async (req, res) => {
  try {
    const { tipo, desde, hasta } = req.query;
    const pool = await poolPromise;

    let tabla = "";

    if (tipo === "incidentes") tabla = "Incidentes";
    else if (tipo === "caidas") tabla = "Caidas";
    else if (tipo === "simulacros") tabla = "Simulacros";
    else return res.status(400).send("Tipo inválido");

    const result = await pool.request().query(`SELECT * FROM ${tabla}`);

    const wb = new ExcelJS.Workbook();
    const ws = wb.addWorksheet("Reporte");

    ws.columns = Object.keys(result.recordset[0]).map((key) => ({
      header: key,
      key,
      width: 25,
    }));

    result.recordset.forEach((row) => ws.addRow(row));

    res.setHeader(
      "Content-Disposition",
      `attachment; filename=reporte_${tipo}.xlsx`
    );
    res.setHeader(
      "Content-Type",
      "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
    );

    await wb.xlsx.write(res);
    res.end();
  } catch (error) {
    console.error("🔥 ERROR en exportarExcel:", error);
    res.status(500).send("Error al exportar Excel");
  }
};
