// controllers/dashboardController.js
const { poolPromise } = require("../config/db");

exports.getResumen = async (req, res) => {
  try {
    const pool = await poolPromise;

    // 📊 Tarjetas de resumen
    const totalActivos = await pool.request().query(`SELECT COUNT(*) AS total FROM Activos`);
    const incidentesAbiertos = await pool.request().query(`
      SELECT COUNT(*) AS total FROM Incidentes WHERE estado = 'Abierto'
    `);
    const proximosSimulacros = await pool.request().query(`
      SELECT COUNT(*) AS total 
      FROM Simulacros 
      WHERE fecha_programada >= CAST(GETDATE() AS DATE)
    `);

    // 🔹 Cumplimiento RTO/RPO (según criticidad)
    const cumplimientoQuery = await pool.request().query(`
      SELECT 
        COUNT(*) AS total,
        SUM(CASE WHEN criticidad IN ('Media', 'Baja') THEN 1 ELSE 0 END) AS cumple
      FROM Activos
    `);

    const totalAct = cumplimientoQuery.recordset[0].total;
    const cumplen = cumplimientoQuery.recordset[0].cumple;

    let cumplimiento = 0;
    if (totalAct > 0) {
      cumplimiento = Math.round((cumplen / totalAct) * 100);
    }

    // 📅 Incidentes por mes
    const incidentesMes = await pool.request().query(`
      SELECT 
        FORMAT(fecha_registro, 'MMM yyyy', 'es-GT') AS mes,
        COUNT(*) AS total
      FROM Incidentes
      GROUP BY FORMAT(fecha_registro, 'MMM yyyy', 'es-GT')
      ORDER BY MIN(fecha_registro)
    `);

    // 🧩 Criticidad de activos
    const criticidad = await pool.request().query(`
      SELECT criticidad, COUNT(*) AS total
      FROM Activos
      GROUP BY criticidad
    `);

    // 🚨 Últimos incidentes
    const recientes = await pool.request().query(`
      SELECT TOP 5 descripcion, FORMAT(fecha_registro, 'dd, MMM') AS fecha
      FROM Incidentes
      ORDER BY fecha_registro DESC
    `);

    // 🗓️ Calendario de simulacros
    const simulacros = await pool.request().query(`
      SELECT 
        tipo_simulacro,
        FORMAT(fecha_programada, 'dd/MM/yyyy') AS fecha_programada
      FROM Simulacros
      ORDER BY fecha_programada ASC
    `);

    res.json({
      resumen: {
        totalActivos: totalActivos.recordset[0].total,
        incidentesAbiertos: incidentesAbiertos.recordset[0].total,
        proximosSimulacros: proximosSimulacros.recordset[0].total,
        cumplimiento
      },
      incidentesMes: incidentesMes.recordset,
      criticidad: criticidad.recordset,
      recientes: recientes.recordset,
      simulacros: simulacros.recordset
    });

  } catch (error) {
    console.error("❌ Error al obtener resumen del dashboard:", error);
    res.status(500).send("Error al obtener dashboard");
  }
};
