// controllers/caidasController.js
const { poolPromise } = require("../config/db");

// =====================================================
// 📌 Obtener todas las caídas (con id_activo y duración legible)
// =====================================================
exports.getCaidas = async (req, res) => {
  try {
    const pool = await poolPromise;
    const result = await pool.request().query(`
      SELECT 
        c.id_caida,
        a.id_activo,         -- ⭐ NECESARIO para editar
        a.nombre_activo,
        c.fecha_inicio,
        c.fecha_fin,
        c.duracion_min,
        CONCAT(
          FLOOR(c.duracion_min / 60), 'h ',
          c.duracion_min % 60, 'm'
        ) AS duracion_legible,
        c.causa,
        c.estado
      FROM Caidas c
      INNER JOIN Activos a ON c.id_activo = a.id_activo
      ORDER BY c.fecha_inicio DESC
    `);

    res.json(result.recordset);
  } catch (error) {
    console.error("❌ Error al obtener caídas:", error);
    res.status(500).send(error);
  }
};

// =====================================================
// ➕ Registrar nueva caída (con duración calculada seguro)
// =====================================================
exports.createCaida = async (req, res) => {
  try {
    const { id_activo, fecha_inicio, fecha_fin, causa, estado } = req.body;

    const pool = await poolPromise;
    await pool.request()
      .input("id_activo", id_activo)
      .input("fecha_inicio", fecha_inicio)
      .input("fecha_fin", fecha_fin || null)   // ⭐ evita error cuando viene ""
      .input("causa", causa)
      .input("estado", estado)
      .query(`
        INSERT INTO Caidas (id_activo, fecha_inicio, fecha_fin, duracion_min, causa, estado)
        VALUES (
          @id_activo,
          @fecha_inicio,
          @fecha_fin,
          CASE 
            WHEN @fecha_fin IS NULL THEN NULL        -- ⭐ evita DATEDIFF con NULL
            ELSE DATEDIFF(MINUTE, @fecha_inicio, @fecha_fin)
          END,
          @causa,
          @estado
        )
      `);

    res.json({ message: "Caída registrada correctamente" });
  } catch (error) {
    console.error("❌ Error al registrar caída:", error);
    res.status(500).send(error);
  }
};

// =====================================================
// ✏️ Actualizar caída (con duración segura)
// =====================================================
exports.updateCaida = async (req, res) => {
  try {
    const { id } = req.params;
    const { id_activo, fecha_inicio, fecha_fin, causa, estado } = req.body;

    const pool = await poolPromise;

    await pool.request()
      .input("id", id)
      .input("id_activo", id_activo)
      .input("fecha_inicio", fecha_inicio)
      .input("fecha_fin", fecha_fin || null)
      .input("causa", causa)
      .input("estado", estado)
      .query(`
        UPDATE Caidas
        SET 
          id_activo = @id_activo,
          fecha_inicio = @fecha_inicio,
          fecha_fin = @fecha_fin,
          duracion_min = CASE
            WHEN @fecha_fin IS NULL THEN NULL
            ELSE DATEDIFF(MINUTE, @fecha_inicio, @fecha_fin)
          END,
          causa = @causa,
          estado = @estado
        WHERE id_caida = @id
      `);

    res.json({ message: "Caída actualizada correctamente" });
  } catch (error) {
    console.error("❌ Error al actualizar caída:", error);
    res.status(500).send(error);
  }
};


// =====================================================
// 🗑️ Eliminar caída
// =====================================================
exports.deleteCaida = async (req, res) => {
  try {
    const { id } = req.params;
    const pool = await poolPromise;

    await pool.request()
      .input("id", id)
      .query("DELETE FROM Caidas WHERE id_caida = @id");

    res.json({ message: "Caída eliminada correctamente" });
  } catch (error) {
    console.error("❌ Error al eliminar caída:", error);
    res.status(500).send(error);
  }
};
