// controllers/activosController.js
const { poolPromise } = require('../config/db');
const sql = require('mssql');

// ✅ Helper: convierte "3" -> "03:00:00" (formato TIME válido para SQL)
function toSQLTime(value) {
  if (value === null || value === undefined || value === '') {
    return '00:00:00';
  }

  // nos quedamos solo con números
  const num = parseInt(String(value).replace(/\D/g, ''), 10);

  if (isNaN(num) || num < 0) {
    return '00:00:00';
  }

  const h = String(num).padStart(2, '0'); // "3" -> "03"
  return `${h}:00:00`; // "03:00:00"
}

// ✅ Obtener todos los activos
const obtenerActivos = async (_req, res) => {
  try {
    const pool = await poolPromise;
    const result = await pool.request().query('SELECT * FROM Activos');
    res.json(result.recordset);
  } catch (err) {
    console.error('Error al obtener activos:', err);
    res.status(500).send('Error al obtener los activos');
  }
};

// ✅ Crear un nuevo activo
const crearActivo = async (req, res) => {
  try {
    console.log('🔥 BODY RECIBIDO EN CREAR:', req.body);

    const {
      nombre_activo,
      tipo,
      descripcion,
      criticidad,
      rto,
      rpo,
      estado,
      id_responsable,
    } = req.body;

    const rtoSQL = toSQLTime(rto);
    const rpoSQL = toSQLTime(rpo);

    const pool = await poolPromise;

    await pool.request()
      .input('nombre_activo', sql.VarChar(100), nombre_activo)
      .input('tipo', sql.VarChar(30), tipo)
      .input('descripcion', sql.VarChar(255), descripcion || null)
      .input('criticidad', sql.VarChar(10), criticidad)
      // ⚠️ usamos VarChar(8) y dejamos que SQL Server lo convierta al tipo TIME
      .input('rto', sql.VarChar(8), rtoSQL)
      .input('rpo', sql.VarChar(8), rpoSQL)
      .input('estado', sql.VarChar(15), estado || 'Activo')
      .input(
        'id_responsable',
        sql.Int,
        id_responsable ? parseInt(id_responsable, 10) : null
      )
      .query(`
        INSERT INTO Activos
        (nombre_activo, tipo, descripcion, criticidad, rto, rpo, estado, id_responsable)
        VALUES (@nombre_activo, @tipo, @descripcion, @criticidad, @rto, @rpo, @estado, @id_responsable)
      `);

    res.status(201).json({ message: 'Activo creado correctamente ✅' });
  } catch (err) {
    console.error('❌ Error al crear activo:', err);
    res.status(500).send(err.message || 'Error al crear el activo');
  }
};

// ✅ Actualizar un activo
const actualizarActivo = async (req, res) => {
  const { id } = req.params;
  const {
    nombre_activo,
    tipo,
    descripcion,
    criticidad,
    rto,
    rpo,
    estado,
    id_responsable,
  } = req.body;

  try {
    console.log('🔥 BODY RECIBIDO EN ACTUALIZAR:', req.body);

    const rtoSQL = toSQLTime(rto);
    const rpoSQL = toSQLTime(rpo);

    const pool = await poolPromise;

    await pool.request()
      .input('id_activo', sql.Int, parseInt(id, 10))
      .input('nombre_activo', sql.VarChar(100), nombre_activo)
      .input('tipo', sql.VarChar(30), tipo)
      .input('descripcion', sql.VarChar(255), descripcion || null)
      .input('criticidad', sql.VarChar(10), criticidad)
      .input('rto', sql.VarChar(8), rtoSQL)
      .input('rpo', sql.VarChar(8), rpoSQL)
      .input('estado', sql.VarChar(15), estado || 'Activo')
      .input(
        'id_responsable',
        sql.Int,
        id_responsable ? parseInt(id_responsable, 10) : null
      )
      .query(`
        UPDATE Activos
        SET nombre_activo  = @nombre_activo,
            tipo           = @tipo,
            descripcion    = @descripcion,
            criticidad     = @criticidad,
            rto            = @rto,
            rpo            = @rpo,
            estado         = @estado,
            id_responsable = @id_responsable
        WHERE id_activo = @id_activo
      `);

    res.json({ message: 'Activo actualizado correctamente ✏️' });
  } catch (err) {
    console.error('❌ Error al actualizar activo:', err);
    res.status(500).send(err.message || 'Error al actualizar el activo');
  }
};

// ✅ Eliminar un activo
const eliminarActivo = async (req, res) => {
  const { id } = req.params;

  try {
    const pool = await poolPromise;
    const result = await pool.request()
      .input('id_activo', sql.Int, parseInt(id, 10))
      .query('DELETE FROM Activos WHERE id_activo = @id_activo');

    if (result.rowsAffected[0] === 0) {
      return res.status(404).json({ message: 'Activo no encontrado' });
    }

    res.json({ message: 'Activo eliminado correctamente 🗑️' });
  } catch (err) {
    console.error('❌ Error al eliminar activo:', err);
    res.status(500).send(err.message || 'Error al eliminar el activo');
  }
};

module.exports = {
  obtenerActivos,
  crearActivo,
  actualizarActivo,
  eliminarActivo,
};
